package gov.va.med.mhv.rxrefill.web.controller;

import java.io.IOException;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.Resource;
import javax.el.ValueExpression;
import javax.faces.application.FacesMessage;
import javax.faces.context.FacesContext;
import javax.faces.event.ComponentSystemEvent;
import javax.portlet.PortletRequest;
import javax.portlet.PortletSession;
import javax.servlet.http.HttpServletRequest;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.exception.SystemException;
import com.liferay.portal.kernel.util.ParamUtil;
import com.liferay.portal.kernel.util.PropsUtil;
import com.liferay.portal.model.User;
import com.liferay.portal.util.PortalUtil;

import gov.va.med.mhv.common.api.dto.InPersonAuthenticationDTO;
import gov.va.med.mhv.common.api.dto.PatientDTO;
import gov.va.med.mhv.common.api.dto.UserProfileDTO;
import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.rxrefill.dto.MedicationHistoryDTO;
import gov.va.med.mhv.rxrefill.dto.PrescriptionDTO;
import gov.va.med.mhv.rxrefill.exception.MHVRuntimeException;
import gov.va.med.mhv.rxrefill.service.MedicationSupplementService;
import gov.va.med.mhv.rxrefill.service.PharmacyRefillWebService;
import gov.va.med.mhv.rxrefill.web.util.RxMessagesUtil;
import gov.va.med.mhv.usermgmt.service.UserMgmtService;

public abstract class AbstractMedicationsController implements Serializable {

	private static final long serialVersionUID = 3021954403145251612L;

	private static Logger log = LogManager.getLogger(AbstractMedicationsController.class);
	
	protected static final String LAST_UPDATED_DATE_FORMAT = "MM/dd/yyyy 'at' HHmm";
	protected static final String ERR_PRCS_RQST = "Error Processing request";
	protected static final String MED_SUPPLEMENT_SUMMARY_PAGE = "medicationSupplementsSummary";
	protected static final String MED_SUPPLEMENT_DETAIL_PAGE = "medicationSupplementDetail";
	protected static final String MED_SUPPLEMENT_ADD_PAGE = "addMedicationSupplements";
	protected static final String MED_SUPPLEMENT_EDIT_PAGE = "editMedicationSupplements";
	protected static final String MED_SUPPLEMENT_DELETE_PAGE = "deleteMedicationSupplements";
	protected static final String MED_SUPPLEMENT_ADD_NEW_PAGE = "saveAndNewMedicationSupplement";
	
	protected static final String MED_VA_MEDICATION_SUMMARY_PAGE="vaMedicationsSummary";
	protected static final String MED_VA_MEDICATION_DETAIL_PAGE="vaMedicationDetail";
	
	private static final String USERPROFILE_DTO_KEY = "LIFERAY_SHARED_userprofiledto";
	private static final int WARN_ON_REPEAT_COUNT = 3;
	
	@Value("${logoutUrl}")
	private String logoutUrl;

	@Value("${appContextroot}")
	private String appContextRoot;
	
	@Value("${rxRefreshThreshold}")
	private Long rxRefreshThreshold;	
	
	@Resource(name="userMgmtServiceProxy")
	protected UserMgmtService userMgmtService;
	
	@Resource(name = "medSupServiceServiceProxy")
	private MedicationSupplementService medicationSupplementService;
	
	@Resource(name = "rxRefillServiceServiceProxy")
	private PharmacyRefillWebService rxRefillWebService;
	
	@Autowired
	private ObjectMapper mapper;
	
	@Autowired
	private RxMessagesUtil rxMessagesUtil;	
	
	protected String firstName;
	protected String lastName;
	protected List<MedicationHistoryDTO> medications;
	protected String medicationSupplementReturnPage;
	protected String vaMedicationReturnPage;

	protected Boolean saveMessage = false;
	protected Boolean addNewMessage = false;
	protected Boolean deleteMessage = false;
	
	protected ValueExpression sortColumn;
	protected String sortBy;
	protected int rowsPerPage;
	UserProfileDTO userProfile;
	private String startAtAddPage="";
	
	// session scoped - concurrenct access monitor object.
	protected final AtomicInteger monitor = new AtomicInteger(0);
	protected Long lastRefreshTime = 0L;
	
	abstract protected List<MedicationHistoryDTO> loadMedications(String userName);

	protected String getHomeURL() {
		return logoutUrl;
	}

	public String getAppContextRoot() {
		return appContextRoot;
	}
	
	public String authorize(ComponentSystemEvent event) throws IOException, PortalException, SystemException {
		User user = null;
		
		try {
			PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
			FacesContext context = FacesContext.getCurrentInstance();
			
			if (request.getUserPrincipal() == null) {
				context.getApplication().getNavigationHandler().handleNavigation(context, null, "rxUnauthorized.xhtml");
			} else {
				
				if (!context.isPostback()) {
					resetMessages();
					setRowsPerPage(10);
				}else {
					saveSortInfo();
				}
				
				// get logged-in user screen name
				PortletSession session = request.getPortletSession();
				String userScreenName = (String) session.getAttribute("LIFERAY_SHARED_userid", PortletSession.APPLICATION_SCOPE);
				
				if (log.isDebugEnabled()) {
					log.debug("user exists " + userScreenName);
				}
				
				user = PortalUtil.getUser(request);
				
				if (user != null) {
					firstName = user.getFirstName();
					lastName = user.getLastName();
				}
				if (!context.isPostback()) {		
					monitor.incrementAndGet();
					
					// Synchronize the Prescription Refresh operation
					synchronized(monitor) {
						try {
							if(monitor.get() >  WARN_ON_REPEAT_COUNT) {
								System.out.println("AtomicInteger value in AbstractMedicationsController is incremented and warn on repeat count value");
								FacesContext.getCurrentInstance().addMessage(null, new FacesMessage(FacesMessage.SEVERITY_ERROR, 
										rxMessagesUtil.getWarnMultipleRefreshes(), rxMessagesUtil.getWarnMultipleRefreshes()));
							}
							
							if((System.currentTimeMillis() - this.lastRefreshTime) >  this.rxRefreshThreshold) {
								System.out.println("lastRefreshTime in AbstractMedicationsController: " + lastRefreshTime + " ;rxRefreshThreshold value: " + this.rxRefreshThreshold);								
							    this.lastRefreshTime = System.currentTimeMillis();
								loadMedications(userScreenName);
							}
						} finally {
							monitor.decrementAndGet();
						}
					}					
					
				}
			}
						
		
			String currentCompleteUrl = PortalUtil.getCurrentURL(request);
			HttpServletRequest httprequest = PortalUtil.getHttpServletRequest(request);
			startAtAddPage = PortalUtil.getOriginalServletRequest(httprequest).getParameter("startAtAddPage"); 
			
			
			if(startAtAddPage != null && startAtAddPage.equals("true")) {
			//if(currentCompleteUrl.contains("startAtAddPage=true")) {
				context.getApplication().getNavigationHandler().
		           handleNavigation(context, null, "/views/rx-refill/addMedicationSupplements.xhtml");
			}
		
		} catch (Exception e) {
			log.error("Error in authorize", e);
			FacesContext.getCurrentInstance().addMessage(null, new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}
		
		
		startAtAddPage="false";
		
		return "";
	}
	
	public String moveToAddPage() {
		return MED_SUPPLEMENT_ADD_PAGE;
	}

	protected  abstract void saveSortInfo();
	
	
	/**
	 * get userProfile for userId
	 * 
	 * @param userId
	 * @return
	 */
	protected UserProfileDTO getCurrentUserProfile(String userName) {
		UserProfileDTO userProfileDto = null;

		try {
			
			PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
			
			// get logged-in user screen name
			String patientStr =  (String)request.getPortletSession().getAttribute(USERPROFILE_DTO_KEY, PortletSession.APPLICATION_SCOPE);
			if(patientStr != null) {
				userProfileDto = mapper.readValue(patientStr, UserProfileDTO.class);
			}
			else {
				userProfileDto = this.userMgmtService.getUserProfile(userName);
			}
			
		} catch (Exception e) {
			log.error("Error in fetching user data " , e);
			throw new MHVRuntimeException(e);
		} 

		return userProfileDto;
	}

	

	/**
	 * get InPersonAuthentication for userProfileId
	 * 
	 * @param userProfileId
	 * @return
	 */
	protected InPersonAuthenticationDTO getIPAPatientById(PatientDTO patientDto) {
		InPersonAuthenticationDTO ipaDto = null;

		try {
			if (null != patientDto) {
				Set<InPersonAuthenticationDTO> ipas = patientDto.getIpas();
				if (null != ipas && ipas.size() > 0 && null!= ipas.iterator() && ipas.iterator().hasNext()) {
					ipaDto = ipas.iterator().next();
				}
				if(ipaDto != null){
				   ipaDto.setPatient(patientDto);
				}
			} else {
				log.error("patient is null");
				throw new MHVRuntimeException("patient is null");
			}
			
		} catch (Exception e) {
			log.error("Error in fetching ipa data " , e);
			throw new MHVRuntimeException(e);
		}

		return ipaDto;
	}
	
	
	/**
	 * get userScreenName from session
	 * 
	 * @return
	 */
	protected String getUserScreenName() {
		PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
		PortletSession session = request.getPortletSession();
		String userScreenName = (String) session.getAttribute("LIFERAY_SHARED_userid", PortletSession.APPLICATION_SCOPE);

		return userScreenName;
	}
	
	
	protected Long getUserProfileId() {
		Long userprofileId = null;
		PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
		PortletSession session = request.getPortletSession();
		Object profileId = session.getAttribute("LIFERAY_SHARED_userprofileid", PortletSession.APPLICATION_SCOPE);
		if (profileId != null) {
			userprofileId = (Long) profileId;
		}
		return userprofileId;
	}
	
	
	public String getMhvContext(){
		String context = PropsUtil.get("mhv.context");
		if (log.isDebugEnabled()) {
			log.debug("getMhvContext() : " + context);
		}
		return context;
    }

	protected String getFormattedDate(Date date) {
		if (date == null) return "";
		SimpleDateFormat df = new SimpleDateFormat(LAST_UPDATED_DATE_FORMAT);
		String formattedDate = df.format(date);
		return formattedDate;
	}
	
	
	protected boolean checkRefillable(PrescriptionDTO rx) {
		if (null != rx.getIsRefillable() && rx.getIsRefillable().booleanValue())
			return true;
		else
			return false;
	}
	
	protected void resetMessages(){
		saveMessage = false;
		addNewMessage = false;
		deleteMessage = false;
	}
	
	protected void processErrorMessages(ResponseUtil response) {
		log.info("isFailure : " + response.isFailure());
		// response has errors
		if (null != response && response.isFailure()) {
			Map<String, String> validationErrors = response.getValidationErrors();

			if (log.isDebugEnabled()) {
				int size = null != validationErrors ? validationErrors.size() : 0;
				log.debug("validationErrors.size() : " + size);
			}

			// check if there are any validation errors in response object
			if (null != validationErrors && validationErrors.size() > 0) {

				for (Map.Entry<String, String> err : validationErrors.entrySet()) {
					if (log.isDebugEnabled()) {
						if (null != err) {
							log.debug("entry.getKey() : " + err.getKey());
							log.debug("entry.getValue() : " + err.getValue());
						}
					}
					FacesContext.getCurrentInstance().addMessage(null, new FacesMessage(FacesMessage.SEVERITY_ERROR, err.getValue(), err.getValue()));
				}
			}

			// check for non-validation errors in response object
			if (null != response.getFailureMessage() && response.getFailureMessage().length() > 0) {
				if (log.isDebugEnabled()) {
					log.debug("FailureMessage : " + response.getFailureMessage());
				}

				FacesContext.getCurrentInstance().addMessage(null, new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
	}
	
	
	public void findUser() {
		PortletRequest   request = (PortletRequest)FacesContext.getCurrentInstance().getExternalContext().getRequest();
		User user=null;
        try{
        	user = PortalUtil.getUser(request);
        }catch(Exception e){
        	
        }
		if(user != null){
			setFirstName(user.getFirstName());
			setLastName(user.getLastName());
		}
	}
	
	public String getLastName() {
		return lastName;
	}
	
	public void setLastName(String lastName) {
		this.lastName = lastName;
	}
	
	public String getFirstName() {
		return firstName;
	}
	
	public void setFirstName(String firstName) {
		this.firstName = firstName;
	}
	
	public MedicationSupplementService getMedicationSupplementService() {
		return medicationSupplementService;
	}
	
	public void setMedicationSupplementService(
			MedicationSupplementService medicationSupplementService) {
		this.medicationSupplementService = medicationSupplementService;
	}
	
	public PharmacyRefillWebService getRxRefillWebService() {
		return rxRefillWebService;
	}

	public void setRxRefillWebService(PharmacyRefillWebService rxRefillWebService) {
		this.rxRefillWebService = rxRefillWebService;
	}

	public List<MedicationHistoryDTO> getMedications() {
		return medications;
	}
	
	public void setMedications(List<MedicationHistoryDTO> medications) {
		this.medications = medications;
	}
	
	public Boolean getSaveMessage() {
		return saveMessage;
	}
	
	public void setSaveMessage(Boolean saveMessage) {
		this.saveMessage = saveMessage;
	}
	
	public Boolean getDeleteMessage() {
		return deleteMessage;
	}
	
	public void setDeleteMessage(Boolean deleteMessage) {
		this.deleteMessage = deleteMessage;
	}
	
	public Boolean getAddNewMessage() {
		return addNewMessage;
	}
	
	public void setAddNewMessage(Boolean addNewMessage) {
		this.addNewMessage = addNewMessage;
	}

	public ValueExpression getSortColumn() {
		return sortColumn;
	}

	public void setSortColumn(ValueExpression sortColumn) {
		this.sortColumn = sortColumn;
	}

	public String getSortBy() {
		return sortBy;
	}

	public void setSortBy(String sortBy) {
		this.sortBy = sortBy;
	}

	public int getRowsPerPage() {
		return rowsPerPage;
	}

	public void setRowsPerPage(int rowsPerPage) {
		this.rowsPerPage = rowsPerPage;
	}

	public UserProfileDTO getUserProfile() {
		return userProfile;
	}

	public void setUserProfile(UserProfileDTO userProfile) {
		this.userProfile = userProfile;
	}
	
	public String getStartAtAddPage() {
		return startAtAddPage;
	}

	public void setStartAtAddPage(String startAtAddPage) {
		this.startAtAddPage = startAtAddPage;
	}
	
}
